<?php

namespace App\Filament\Resources\Receipts\Schemas;

use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Schema;

class ReceiptForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                TextInput::make('customer_id')
                    ->label(__('Customer Id'))
                    ->required()
                    ->unique(ignoreRecord: true)
                    ->maxLength(20)
                    ->live(debounce: 500) // ⏱ wait 0.5s after typing stops
                    ->rule('regex:/^[A-Z0-9]+$/')
                    ->afterStateUpdated(function ($state, callable $set) {
                        if (filled($state)) {
                            $set('customer_id', strtoupper($state));
                        }
                    }),
                 DatePicker::make('receipt_date')
                    ->label('Receipt Date')
                    ->required(),
                TextInput::make('invested_amount')
                    ->label('Invested Amount')
                    ->numeric()
                    ->required(),
                Select::make('company_name')
                    ->label('Company Name')
                    ->required()
                    ->options([
                        'Thangam Finance' => 'Thangam Finance',
                        'Golden Trust' => 'Golden Trust',
                        'SriDhan Investments' => 'SriDhan Investments',
                    ])
                    ->searchable(),
                TextInput::make('receipt_customer_name')
                    ->label('Receipt Customer Name')
                    ->required()
                    ->live(debounce: 500) // ⏱ wait 0.5s after typing stops
                    ->extraAttributes([
                        'onkeypress' => "return /[A-Za-z\\s]/.test(event.key);",
                    ])
                    ->afterStateUpdated(function ($state, callable $set) {
                            if (filled($state)) {
                                // Convert to uppercase safely and trim spaces
                                $set('receipt_customer_name', strtoupper($state));
                            }
                        })
                    ->helperText('Only letters allowed'),

                Select::make('relationship')
                    ->label('Relationship')
                    ->required()
                    ->options([
                        'Father' => 'Father',
                        'Mother' => 'Mother',
                        'Brother' => 'Brother',
                        'Sister' => 'Sister',
                        'Friend' => 'Friend',
                        'Self' => 'Self',
                        'Other' => 'Other',
                    ]),
                FileUpload::make('receipt_image')
                    ->label('Receipt Image')
                    ->directory('receipts')
                    ->image()
                    ->acceptedFileTypes(['image/jpeg', 'image/jpg']) // ✅ browser filter
                    ->rules([
                        'mimes:jpg,jpeg',
                        'max:512', // 512 KB
                    ])
                    ->helperText('Only JPG/JPEG, max 500 KB'),
            ]);
    }
}
