<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use App\Models\User;

class ImportUsersFromCsv extends Command
{
    protected $signature = 'import:users {file=storage/app/import/users.csv}';
    protected $description = 'Import users from a CSV file into the users table';

    private function convertDate($date)
    {
        if (empty($date)) {
            return null;
        }

        // Try to handle formats like 24/12/1992 or 24-12-1992
        $date = str_replace(['.', '-', '/'], '-', trim($date));

        $parts = explode('-', $date);

        // If it's in DD-MM-YYYY format
        if (count($parts) === 3 && strlen($parts[2]) === 4) {
            [$day, $month, $year] = $parts;
            if (checkdate((int)$month, (int)$day, (int)$year)) {
                return "{$year}-{$month}-{$day}";
            }
        }

        // Fallback to null if invalid
        return null;
    }

    public function handle()
    {
        $file = base_path($this->argument('file'));

        if (!file_exists($file)) {
            $this->error("File not found: {$file}");
            return;
        }

        $handle = fopen($file, 'r');
        if (!$handle) {
            $this->error("Unable to open file: {$file}");
            return;
        }

        // Skip header
        $header = fgetcsv($handle);

        $count = 0;
        while (($row = fgetcsv($handle)) !== false) {
            // Adjust based on your column order
            [
                $sl_no,
                $vg_member_code,
                $vg_member_name,
                $vg_father_name,
                $vg_address,
                $aadhar_no,
                $pan_no,
                $occupation,
                $dob,
                $age,
                $sex,
                $prime_city,
                $mobile_no,
                $whatsapp_no,
                $email_id,
                $old_cadre,
                $old_cp_code,
                $vg_group_code,
                $vg_group_name,
                $federation_code,
                $federation_name,
            ] = $row;

            // Handle username and password
            $username = $mobile_no;
            $password = substr($mobile_no, 0, 6);

            // Skip empty or invalid mobile numbers
            if (empty($mobile_no)) {
                $this->warn("Skipping empty mobile number for row: {$vg_member_name}");
                continue;
            }

            // Avoid duplicates
            if (User::where('mobile', $mobile_no)->orWhere('email', $email_id)->exists()) {
                $this->warn("Duplicate user skipped: {$mobile_no}");
                continue;
            }

            User::create([
                'name' => $vg_member_name,
                'father_name' => $vg_father_name,
                'address' => $vg_address,
                'aadhar_no' => $aadhar_no,
                'pan_no' => $pan_no,
                'occupation' => $occupation,
                'dob' => $this->convertDate($dob),
                'sex' => $sex,
                'primary_city' => $prime_city,
                'mobile' => $mobile_no,
                'whatsapp_no' => $whatsapp_no,
                'member_level' => $vg_member_code,
                'email' => $email_id,
                'member_cadre' => $old_cadre,
                'member_code' => $old_cp_code,
                'vg_group_code' => $vg_group_code,
                'vg_group_name' => $vg_group_name,
                'federation_code' => $federation_code,
                'federation_name' => $federation_name,
                'username' => $username,
                'password' => Hash::make($password),
                'role' => 'USER',
            ]);

            $count++;
        }

        fclose($handle);

        $this->info("✅ Successfully imported {$count} users!");
    }
}
